#lang rhombus/scribble/manual

@(import:
    pict open
    draw
    meta_label:
      rhombus open
      pict open
      draw
    "pict_eval.rhm" open
    "timeline.rhm" open
    // to sequence examples
    "static-overview.scrbl")

@title(~tag: "animated-pict"){Animated Picts}

An @deftech{animated pict} is based a function that takes a time
@math{t} and produces a @tech{static pict} to draw a that point in time.
In the same way that a static pict needs a @tech{bounding box} to
determine how it draws in combination with other pictures, an animated
pict needs a @deftech{time box} that determines how it synchronizes with
other animated picts. A time box's @deftech{duration} is analogous to a
bounding box's width or height.

@margin_note{See
 @secref(~doc: ModulePath 'lib("rhombus/slideshow/scribblings/rhombus-slideshow.scrbl")',
         "overview")
 for examples of how animated picts work with slide presentations.}

@section{Time Boxes and Durations}

A static pict can be used as a animated pict of duration @rhombus(1)
whose rendering is the static pict during its time box. An animated
pict's rendering is not constrained to its time box, however; just like
a static pict can potentially affect any pixel outside its bounding box
in an infinite drawing plane, an animated pict has a rendering for
all time before and after its time box. A static pict's rendering
outside of its time box is a ghosted version of the static pict (in the
sense of @rhombus(Pict.ghost)).

Using the static rectangle pict @rhombus(rect) from before, the
following diagram shows its rendering at four different points on a
timeline, where @math{t} is relative to the pict's time box, and the
time box is shown in pink on the timeline. Nothing is drawn under
@math{t = -1}, @math{t = 1}, or @math{t = 2} because the pict is ghosted
at those times.

@(def rect = rectangle(~width: 40, ~height: 20, ~fill: "lightblue"))
@(def circ = circle(~size: 40, ~fill: "lightgreen"))

@timeline(rect,
          [-1, 0, 1, 2],
          ~label: "rect"
          ~balance_label: #true,
          ~pre: 1)

An animated pict's rendering in time can be shifted relative to its time
box, just like a static pict's drawing can be shifted relative to its
bounding box. The @rhombus(Pict.time_pad) function extends time box on
either end, which can shift the pict's drawing relative to the start of
its time box. Let's shift @rhombus(rect) to define @rhombus(late_rect):

@examples(  
  ~eval: pict_eval
  ~defn:
    def late_rect = rect.time_pad(~before: 1)
)

Here's the rendering of @rhombus(late_rect) over time, which shows that
its duration is @rhombus(2), but it doesn't become unghosted until
relative time @math{t = 1}:

@timeline(rect.time_pad(~before: 1),
          [-1, 0, 1, 2],
          ~label: "late_rect",
          ~balance_label: #true,
          ~post: 1)

Operations like @rhombus(beside), @rhombus(stack), and
@rhombus(Pict.pad) operate on all picts, both static and animated.
Placing an animated pict @rhombus(beside) another corresponds to
rendering the two picts concurrently and placing each rendered form
@rhombus(beside) the other. Padding an animated pict means padding every
rendering that is generated by the animated pict over time to produce a
new animated pict---one with the same duration as the unpadded original.

Time-box padding plus composite operations like @rhombus(stack) allow
picts to be staged in an animation. If we pad @rhombus(circ)'s time box
on the ``after'' side to define @rhombus(early_circ), then
@rhombus(early_circ) and @rhombus(late_rect) have the same timeboxes, and
combining them with @rhombus(stack) makes a composite pict
@rhombus(ping_pong):

@examples(  
  ~eval: pict_eval
  ~defn:
    def early_circ = circ.time_pad(~after: 1)
    def ping_pong = stack(early_circ, late_rect)
)

Since we padded @rhombus(circ) with @rhombus(~after) and @rhombus(rect)
with @rhombus(~before), the @rhombus(circ) and @rhombus(rect) components
of @rhombus(ping_pong) are unghosted at different times---but their
ghosts preserve the space that they will sometimes fill:

@timeline(stack(circ.time_pad(~after: 1), rect.time_pad(~before: 1)),
          ~label: "ping_pong",
          [-1, 0, 1, 2])

We could have used @rhombus(overlay) instead of @rhombus(stack) to have
the two shapes overlap, and then a circle would appear to change to a
square in place:

@examples(  
  ~eval: pict_eval
  ~defn:
    def together_apart = overlay(early_circ, late_rect)
)

The combined picture would have a bounding box based on combined
bounding boxes, since that what @rhombus(overlay) does. In this picture,
we show each image with a little padding and then a rectangle for the
rulting bound box:

@timeline(rectangle(~around: overlay(circ.time_pad(~after: 1), rect.time_pad(~before: 1)).pad(3)),
          ~label: "together_apart",
          [-1, 0, 1, 2])

The @rhombus(switch) function, in contrast, takes care of shifting
its arguments to make them sequential, and it stops using each pict at
the end of its time box and start using the next. That way, we don't
have to shift manually, and the bounding boxes are independent.

@examples(
  ~eval: pict_eval
  ~defn:
    def one_at_a_time = switch(early_circ, late_rect)
)

@timeline(switch(rectangle(~around: circ.pad(3)), rectangle(~around: rect.pad(3))),
          ~label: "one_at_a_time"
          [-1, 0, 1, 2])

Whether @rhombus(overlay) or @rhombus(switch) is the right choice
depends on the context. The @rhombus(sequential) function can help
with shifting time boxes to make picts sequential without otherwise
combining them.

@section{Epochs}

We have not yet defined the units of time @math{t} for an animated pict.
Animated picts are meant to represent slide presentations, where we want
a notion of time that is compatible with both slide-based stepping and
real-time animations to transition between slides. Thus, an increment of
@rhombus(1) in @math{t} corresponds to advancing the slide, and
increments between @rhombus(0) and @rhombus(1) are transition points
from one slide to the next. More abstractly, we refer to each increment
of @math{t} by @rhombus(1) as an @deftech{epoch}. A pict's duration is
always measured in integer epochs.

When an epoch advances (i.e., when the presenter hits the space bar to
advance the slide), then an epoch-specific animation function receives a sequence of
numbers between @rhombus(0) and @rhombus(1) to represent the
intermediate transition points between the old slide and the new one.
The rate at which those intermediate results are used is based the
epoch's @deftech{extent}, which is measured in seconds. An animated pict
can have a different extent for each of its epochs. A static pict's
epochs all have extent @rhombus(0), meaning that it doesn't need to draw
any transitions on the way out of the epoch.

Suppose that we have a pict whose duration is @rhombus(3) epochs and the
epochs have extents @rhombus(0.5), @rhombus(1.0), and @rhombus(0.5)
seconds, respectively. If we stretch that pict's conceptual timeline to
match real time, then it might look something like this, depending on
how fast the presenter advances each slide:

@centered(stretched_timeline)

The red dots to start epochs become ovals to represent a slide at rest.
The right end end of a red oval is the start of a transition to a new
slide, and the new slide arrives fully at the left edge of the next red
oval. The space between between red ovals for epoch @rhombus(1) is twice
as large as the other spaces, because that epoch's extent is twice as
long. During that middle animation, when the conceptual time is @math{t
 = 1.5}, the animation function associated with the epoch receives the
epoch-relative number @rhombus(0.5) (not a time-box-relative
@rhombus(1.5)).

To create an animated picture with a duration of @rhombus(1) epoch and a
non-@rhombus(0) extent for that epoch, use the @rhombus(animate)
constructor:

@examples(  
  ~eval: pict_eval
  ~defn:
    def fade_out = animate(fun (n): circ.alpha(1-n))
)

@(def fade_out = animate(fun (n): circ.alpha(1-n)))

The @rhombus(fade_out) pict's one epoch in this example uses the
default extent of @rhombus(0.5) seconds, but an extent could have been
specified with an @rhombus(~extent) argument. This pict's animation is
illustrated by the following conceptual timeline, which shows the circle
solid during its slide pause, fading afterward to the next slide, and
ghosted outside its time box:

@timeline(fade_out,
          ~label: "fade_out",
          [-0.5, -0.25, 0, 0.25, 0.5, 0.75, 1, 1.25])

We can combine a fade-in animation with a fade-out animation, and then
shift the pict's time box so that it corresponds to the point where the
pict is fully faded in.

@examples(  
  ~eval: pict_eval
  ~defn:
    def fade_in = animate(fun (n): circ.alpha(n))
    def fade_in_out = switch(fade_in, fade_out).time_pad(~before: -1)
)

@(def fade_in = animate(fun (n): circ.alpha(n)))
@(def fade_in_out = switch(fade_in, fade_out).time_pad(~before: -1))

@timeline(fade_in_out,
          ~label: "fade_in_out",
          [-1, -0.66, -0.33, 0, 0.33, 0.66, 1])

Functions like @rhombus(sequential) and @rhombus(Pict.epoch_set_extent)
support splicing within-epoch animations and fine-grained control over
epoch extents.

@section{Pad versus Sustain}

Suppose that we want our circle pict to appear first, and then our
rectangle pict, but we want the circle to stick around. We could use
@rhombus(switch) to combine two instances of @rhombus(circ), and then
combine that with @rhombus(late_rect):

@examples(
  ~eval: pict_eval
  ~defn:
    def long_circ = switch(circ, circ)
    def together = stack(long_circ, late_rect)
)

@(def long_circ = switch(circ, circ))

@timeline(stack(long_circ, rect.time_pad(~before: 1)),
          ~label: "together",
          [-1, 0, 1, 2])

This strategy is not quite right in general, however. If we use
@rhombus(fade_in_out) instead of @rhombus(circ), then the circle first
fades out, and then it suddenly reappears with the square:

@examples(
  ~eval: pict_eval
  ~defn:
    def two_fade = switch(fade_in_out, fade_in_out)
    def together_try = stack(two_fade, late_rect)
)

@(def two_fade = switch(fade_in_out, fade_in_out))

@timeline(stack(two_fade, rect.time_pad(~before: 1)),
          ~label: "together_try",
          [-0.33, 0, 0.33, 0.66, 1, 1.33, 1.66, 2])

Instead of this behavior, we want to insert epochs before the current
last epoch, where each inserted epoch has a static pict matching the
beginning of the last epoch. This @deftech{sustain} transformation of an
animated pict is needed so often that it is part of the definition of an
animated pict; sustain a pict using @rhombus(Pict.sustain), and the
details of what sustain means for a pict can be customized in an option
to @rhombus(animate). The default in @rhombus(animate) is not what we
want in this case, so we'll create a variant of @rhombus(fade_out) that
sustains at the desired side of the animation:

@(def fade_out2 = animate(fun (n): circ.alpha(1-n), ~sustain_edge: #'before))
@(def fade_in_out2 = switch(fade_in, fade_out2).time_pad(~before: -1))

@examples(  
  ~eval: pict_eval
  ~defn:
    def fade_out2 = animate(fun (n): circ.alpha(1-n),
                            ~sustain_edge: #'before)
    def fade_in_out2 = switch(fade_in, fade_out2).time_pad(~before: -1)
)

In fact, sustaining an animated pict is the right choice so often that
it is the default way that operations like @rhombus(overlay) and
@rhombus(stack) extend the duration of each pict to match others in the
combination. In other words, we don't have to manually force
@rhombus(fade_in_out) and @rhombus(late_rect) to have the same duration,
and @rhombus(stack) will sustain @rhombus(fade_in_out) to make its
duration match @rhombus(late_rect):

@examples(
  ~eval: pict_eval
  ~defn:
    def together_fade = stack(fade_in_out2, late_rect)
)

@(def together_fade = stack(fade_in_out2, rect.time_pad(~before: 1)))

@timeline(together_fade,
          ~label: "together_fade",
          [-0.33, 0, 0.33, 0.66, 1, 1.33, 1.66, 2])

While sustaining is usually the right choice for combining animated
picts, functions like @rhombus(overlay) and @rhombus(stack) accept a
@rhombus(~duration) optional argument to select a mode. The argument can be
@rhombus(#'pad) instead of @rhombus(#'sustain) to normalize
durations by padding instead of sustaining. Sometimes, the choice is better
associated with a pict than at a combination, and pict can be made
nonsustaining through @rhombus(Pict.nonsustaining), which causes
a sustain on the pict to be the same as time padding.

@section{Snapshots and Convenience Accessors}

An @tech{animated pict} does not have a unique bounding box, since its
width, height, and other properties can change over time. Nevertheless,
properties like @rhombus(Pict.width) and @rhombus(Pict.height) can be
access from an animated pict. The result in that case corresponds to a
snapshot of the pict at the start of its time box, i.e., at time
@rhombus(0) within epoch @rhombus(0).

The @rhombus(Pict.snapshot) method explicitly converts an animated pict
to a static pict that shows the animate pict's representation at a
specific epoch and time. The @rhombus(Pict.snapshot) method accepts a
number in the inclusive range @rhombus(Int.in(0, 1), ~annot) for the
time within an epoch, which means that there are two different ways to
get a snapshot at boundary times: @rhombus(1) within epoch
@rhombus(N, ~var) and @rhombus(0) with epoch @rhombus(N-1, ~var). Those
two results can be understood as a result infinitesimally before or
after the instant, allowing both side of a discontinuity in the timeline
(e.g., the point where a static pict becomes a ghost).
