#lang rhombus/scribble/manual
@(import:
    "common.rhm" open
    meta_label:
      rhombus/serialize open)

@title{Serializables}

A @deftech{serializable} value is one of the predefined serializable
datatypes (numbers, strings, lists, etc.; see
@rhombus(Serializable, ~annot)) or an instance of a @tech{class} that is
declared with a @rhombus(serializable, ~class_clause) clause. The
@rhombusmodname(rhombus/serialize) module provides functions for
serializing and deserializing values.

@doc(
  ~nonterminal:
    entry_point: block

  class_clause.macro 'serializable'
  class_clause.macro 'serializable:
                        $option
                        ...'
  grammar option:
    ~version $nonneg_int
    ~version: $nonneg_int
    ~serialize: $entry_point
    ~deserialize: $entry_point
    ~deserialize_shell: $entry_point
    ~deserialize_fill: $entry_point
){

 A @tech{class clause} recognized by @rhombus(class) that makes
 instances of the class work with @rhombus(serialize). When a
 @rhombus(class) has a @rhombus(serializable, ~class_clause), it must be
 declared at the top level of a module or in an interactive context.

 When @rhombus(serializable, ~class_clause) is declared without any
 @rhombus(option)s, the class must declare no uninitialized private
 fields. An instance is serialized by extracting and serializing all of
 its public fields, and it is deserialized by calling the class's
 constructor. The constructor is called using the convention of a default
 constructor, but the class can have a custom constructor that accepts
 the same arguments.

 When @rhombus(serializable, ~class_clause) is declared with
 @rhombus(option)s, each @rhombus(option) must use a distinct keyword to
 customize serialization or deserialization:

@itemlist(

 @item{@rhombus(~version): Determines the name of a @tech{submodule}
  that exports the deserialization function. By incrementing the version
  past its default of @rhombus(0) and explicitly declaring a submodule
  with the name corresponding to an older version, data from older
  serializations can be supported while modifying a class's fields. See
  @rhombus(deserializer, ~decl) for more information about deserializer
  submodules.}

 @item{@rhombus(~serialize): Provides a custom serialization function as
  an @tech{entry point}. The entry point is like a method, and it must
  accept zero arguments; @rhombus(this), field names, etc., are bound
  within the entry point for the object being serialized. The result must
  be an @tech{array} containing the object's content, and the array's
  content will be recursively serialized. The default serializer creates
  an array that has each constructor field's value in the order that the
  fields are declared. A custom serialization entry point must be provided
  if a class has any private fields without default values.}

 @item{@rhombus(~deserialize): Provides a custom deserialization
  function as an @tech{entry point}. The entry point must accept as many
  arguments as elements of the vector produced by the serialization entry
  point. The default deserializer passes the arguments on to the class's
  constructor following the class's default constructor protocol
  (potentially using keyword arguments).}

 @item{@rhombus(~deserialize_shell) and @rhombus(~deserialize_fill):
  Optional deserialization support for mutable objects that may have
  reference cycles. If either of these @rhombus(option)s is specified,
  then @rhombus(~deserialize), @rhombus(~deserialize_shell), and
  @rhombus(~deserialize_fill) must all be specified. The
  @rhombus(~deserialize_shell) @tech{entry point} must accept zero
  arguments and return an object that will serve as the deserialized
  object. The @rhombus(~deserialize_fill) entry point must accept two
  arguments: the object returned by @rhombus(~deserialize_shell)'s entry
  point and an object generated by @rhombus(~deserialize)'s entry point to
  hold the final content; the @rhombus(~deserialize_fill) entry point
  should copy content from the latter to the former, and its result is
  ignored. If @rhombus(~deserialize_shell) and @rhombus(~deserialize_fill)
  are not specified, then objects with reference cycles involving only
  such objects cannot be serialized. The @rhombus(~deserialize_shell) and
  @rhombus(~deserialize_fill) entry points will be used only when
  serialized data contained cycles.}

)

}


@section{Serialization and Deserialization}

@docmodule(rhombus/serialize)

@doc(
  annot.macro 'Serializable'
){

 An annotation (not an interface) that is satisfied by value that
 may be serializable. The @rhombus(Serializable, ~annot) annotation does
 not check that elements within a compound value (such as a list) are
 also serializable, so it is not a complete check for whether
 @rhombus(serialize) will succeed.

 The following pre-defined datatypes are serializable:

@itemlist(

 @item{booleans, numbers, characters, interned symbols, unreadable symbols,
       keywords, strings, byte strings, paths (for a specific convention),
       regexp values, and @rhombus(#:void);}

 @item{lists, arrays, boxes, maps, sets, pairs, and pair lists.}

)

}

@doc(
  fun serialize(v :: Serializable,
                ~out: out :: maybe(Port.Output) = #false)
    :: Bytes || Void
){

 Serializes @rhombus(v) to a @tech{byte string}. If @rhombus(out) is an
 @tech{output port}, then the serialized data is written to @rhombus(out)
 and @rhombus(#void) is returned. Otherwise, the result is a byte string
 containing the serialized data.

 The byte-string encoding produced by @rhombus(serialize) is independent
 of the Racket and Rhombus version, except as future versions introduce
 extensions that are not currently recognized.

}

@doc(
  fun deserialize(in :: Bytes || Port.Input) :: Any
){

 Deserializes value from @rhombus(in), reversing the encoding of
 @rhombus(serialize).

}

@doc(
  ~nonterminal:
    entry_point: block

  decl.macro 'deserializer:
                $option
                ...'

  grammar option:
    ~deserialize: $entry_point
    ~deserialize_shell: $entry_point
    ~deserialize_fill: $entry_point
){

 For use within a @tech{submodule}, defines and exports
 @rhombus(deserialize, ~datum) in the enclosing module.

 The enclosing submodule normally should have a name of the form
 @List(@rhombus(deserialize_, ~datum), @rhombus(Name, ~var), @rhombus(_, ~datum), @rhombus(version, ~var))
 where @rhombus(Name, ~var) is the name of a serializable class declared
 in the submodule's parent, and @rhombus(version, ~var) is the class's
 old version.

 The intent of a @rhombus(deserializer) is that the
 @rhombus(~deserialize) @tech{entry point} accepts the content of an
 old-version object and returns a current-version object corresponding
 the the old object. For example, default values may be used for fields
 added in a newer version of the class.

}
