#lang scribble/manual
@(require (for-label racket version-case))


@title{version-case: conditionally compile code based on current version number}
@author[@author+email["Danny Yoo" "dyoo@hashcollision.org"]
        @author+email["Sam Tobin-Hochstadt" "samth@racket-lang.org"]]

@centered{@smaller{Source code can be found at:
@url{https://github.com/samth/version-case}.}}



This library provides support for conditionally compiling code based
on the version of Racket.  One common application of this module is to
write compatibility bindings.

The library is supported for Racket 6.0 and later.


@section{Example}

The following example shows how one can use
@racketmodname[web-server/dispatchers/dispatch-logresp #:indirect],
a web server logging facility that supports
both request and response logging,
if the current Racket version is at least 8.6,
while falling back to @racketmodname[web-server/dispatchers/dispatch-log #:indirect],
another web server logging facility that only supports request logging,
when the current Racket version is less than 8.6.

@racketblock[
(require version-case
         (prefix-in sequencer: web-server/dispatchers/dispatch-log)
         (for-syntax racket/base))
(version-case
  [(version< (version) "8.6")
   (require (prefix-in log: web-server/dispatchers/dispatch-log))
   (define-syntax-rule (with-logging (options ...) dispatcher ...)
     (sequencer:make (log:make options ...) dispatcher ...))]
  [else
   (require (prefix-in log: web-server/dispatchers/dispatch-logresp))
   (define-syntax-rule (with-logging (options ...) dispatcher ...)
     (log:make options ... (sequencer:make dispatcher ...)))])

...
(serve #:dispatch
       (with-logging ()
         ...))
]
                

Another simple example:
@codeblock|{
#lang racket/base
(require version-case
         (for-syntax racket/base))
(printf "~a~n" (version-case [(version<= (version) "4")
                              'huh?]
                             [else
                              'ok]))}|


@section{Usage}

@defmodule[version-case]

@defform[(version-case clause ...)
         #:grammar
         [(clause [test code ...]
                  [#,(racketid else) code ...])]]
@racket[version-case] is a macro that expands out to one of the
@racket[code] blocks, depending on which test succeeds first.  The
@racket[test] expression is evaluated at compile-time.  Some
version-comparing functions are available for convenience.

@defproc[(version< [v1 string?] [v2 string?]) boolean?]{
Returns true if @racket[v1] is less than @racket[v2].
}
@defproc[(version<= [v1 string?] [v2 string?]) boolean?]{
Returns true if @racket[v1] is less than or equal to @racket[v2].
}
@defproc[(version= [v1 string?] [v2 string?]) boolean?]{
Returns true if @racket[v1] is equal to @racket[v2].
}
@defproc[(version> [v1 string?] [v2 string?]) boolean?]{
Returns true if @racket[v1] is greater than @racket[v2].
}
@defproc[(version>= [v1 string?] [v2 string?]) boolean?]{
Returns true if @racket[v1] is greater than or equal to @racket[v2].
}





@section{Gotchas}

The tests are done at compile time.  If the language of your module
doesn't include compile-time bindings for function application, you
may see funny error messages.  For example, @racket[racket/base]
doesn't automatically provide the necessary compile-time bindings, so
if you use @racket[version-case] with it, you also need to do a
@racket[(require (for-syntax racket/base))].



@section{Thanks}

Special thanks to Carl Eastlund providing the implementation that
doesn't use @racket[eval], and for feedback.  Thanks also to Ambjorn
Elder for noticing a bug regarding the use of syntax-case within a
version-case's body.
