#lang racket/base

(require racket/contract
         racket/list
         racket/match)

(provide/contract [version<= (string? string? . -> . boolean?)]
                  [version< (string? string? . -> . boolean?)]
                  [version= (string? string? . -> . boolean?)]
                  [version>= (string? string? . -> . boolean?)]
                  [version> (string? string? . -> . boolean?)])


;; The definitions of racket-version, string->version, and
;; version<= were adapted (copied and pasted) from PLaneT's
;; implementation in (planet/private/planet-shared.ss).

(struct racket-version (numbers) #:transparent)

;; string->version : string -> racket-version | #f
(define (string->version str)
  (match str
    ;; Old style numbering (with three digits in front)
    [(regexp #rx"^([0-9][0-9][0-9])([.0-9]*)$"
             (list _
                   (app string->number major)
                   after-major))

     (define-values (minor maintenances)
       (match (map string->number (rest (regexp-split "\\." after-major)))
         ['() (values 0 '())]
         [(cons v vs) (values v vs)]))

     (racket-version (list*
                      (remainder (quotient major 100) 10)
                      (remainder (quotient major 10) 10)
                      (remainder major 10)
                      minor
                      maintenances))]
    ;; New style numbering
    [(regexp #rx"^([.0-9]*)$" (list _ ver))
     (racket-version (map string->number (regexp-split "\\." ver)))]

    [_ #f]))

;; version-cmp: racket-version racket-version -> (union -1 0 1)
;; Returns -1 if v1 < v2, 0 if v1 = v2, and 1 if v1 > v2.
(define (version-cmp v1 v2)
  (let loop ([v1 (racket-version-numbers v1)]
             [v2 (racket-version-numbers v2)])
    (match* (v1 v2)
      [('() '()) 0]
      [('() (cons _ _)) -1]
      [((cons _ _) '()) 1]
      [((cons a as) (cons b bs))
       (cond
         [(= a b) (loop as bs)]
         [(< a b) -1]
         [else 1])])))


;; version<= : string string -> boolean
;; determines if a is the version string of an earlier
;; mzscheme release than b
;; [n.b. this relies on a guarantee from Matthew that
;; mzscheme version x1.y1 is older than version x2.y2 iff
;;  x1<x2 or x1=x2 and y1<y2]
(define (version<= a b)
  (let ([a (string->version a)]
        [b (string->version b)])
    (not (= (version-cmp a b)
            1))))

(define (version>= a b)
  (let ([a (string->version a)]
        [b (string->version b)])
    (not (= (version-cmp a b)
            -1))))

(define (version= a b)
  (let ([a (string->version a)]
        [b (string->version b)])
    (= (version-cmp a b)
       0)))

(define (version< a b)
  (let ([a (string->version a)]
        [b (string->version b)])
    (= (version-cmp a b)
       -1)))

(define (version> a b)
  (let ([a (string->version a)]
        [b (string->version b)])
    (= (version-cmp a b)
       1)))
